<?php
/*
Plugin Name: Sity Cloud Firewall Protection
Description: إضافة حماية بسيطة من محاولات الضغط الزائد (DDoS-like) لمواقع ووردبريس (خاصة بيئة Multisite) مع إشعار حماية باسم Sity Cloud.
Version: 1.1
Author: Sity Cloud
*/

/**
 * إضافة صفحة إعدادات الحماية في لوحة تحكم الموقع
 */
add_action('admin_menu', 'sityc_firewall_menu');
function sityc_firewall_menu() {
    add_menu_page(
        'حماية Sity Cloud',       // Page title
        'حماية Sity Cloud',       // Menu title
        'manage_options',         // Capability
        'sityc-firewall',         // Menu slug
        'sityc_firewall_page',    // Callback function
        'dashicons-shield',       // Icon
        6                         // Position
    );
}

/**
 * صفحة إعدادات الحماية
 */
function sityc_firewall_page() {

    // حفظ الإعدادات
    if (isset($_POST['sityc_firewall_save_settings'])) {
        check_admin_referer('sityc_firewall_save', 'sityc_firewall_nonce');

        $status       = isset($_POST['sityc_firewall_status']) ? sanitize_text_field($_POST['sityc_firewall_status']) : 'off';
        $request_limit = isset($_POST['sityc_firewall_request_limit']) ? intval($_POST['sityc_firewall_request_limit']) : 20;

        if ($request_limit < 5) {
            $request_limit = 5; // حد أدنى معقول
        }

        update_option('sityc_firewall_status', $status);
        update_option('sityc_firewall_request_limit', $request_limit);

        echo '<div class="notice notice-success is-dismissible"><p>تم تحديث إعدادات الحماية بنجاح ✅</p></div>';
    }

    // زر مسح IPs المحظورة
    if (isset($_POST['sityc_firewall_reset_ips'])) {
        check_admin_referer('sityc_firewall_reset_ips', 'sityc_firewall_reset_nonce');
        update_option('sityc_blocked_ips', []);
        echo '<div class="notice notice-info is-dismissible"><p>تم مسح قائمة عناوين IP المحظورة.</p></div>';
    }

    // جلب القيم الحالية
    $current_status  = get_option('sityc_firewall_status', 'off');
    $request_limit   = intval(get_option('sityc_firewall_request_limit', 20));
    if ($request_limit < 5) {
        $request_limit = 20;
    }
    $blocked_ips     = get_option('sityc_blocked_ips', []);
    if (!is_array($blocked_ips)) {
        $blocked_ips = [];
    }
    ?>
    <div class="wrap sityc-firewall-wrap">
        <h1 class="sityc-title">🛡️ حماية المواقع - Sity Cloud Firewall</h1>

        <div class="sityc-card">
            <h2>إعدادات الحماية</h2>
            <p class="sityc-description">
                من هنا تقدر تشغل / توقف الحماية، وتحدد الحد الأقصى لعدد الطلبات المسموح بها لكل جلسة قبل حظر الـ IP.
            </p>

            <form method="post" action="">
                <?php wp_nonce_field('sityc_firewall_save', 'sityc_firewall_nonce'); ?>

                <div class="sityc-status-container">
                    <label class="sityc-radio">
                        <input type="radio" name="sityc_firewall_status" value="on" <?php checked($current_status, 'on'); ?>>
                        <span class="sityc-radio-label sityc-on">تشغيل الحماية</span>
                    </label>
                    <label class="sityc-radio">
                        <input type="radio" name="sityc_firewall_status" value="off" <?php checked($current_status, 'off'); ?>>
                        <span class="sityc-radio-label sityc-off">إيقاف الحماية</span>
                    </label>
                </div>

                <div class="sityc-field-group">
                    <label for="sityc_firewall_request_limit">عدد الطلبات المسموح بها لكل جلسة قبل الحظر:</label>
                    <input type="number" min="5" id="sityc_firewall_request_limit" name="sityc_firewall_request_limit" value="<?php echo esc_attr($request_limit); ?>" />
                    <p class="sityc-help">مثال: 20 يعني بعد 20 طلب في نفس الجلسة، يتم حظر الـ IP مؤقتًا.</p>
                </div>

                <button type="submit" name="sityc_firewall_save_settings" class="button button-primary sityc-btn-primary">
                    💾 حفظ الإعدادات
                </button>
            </form>
        </div>

        <div class="sityc-card">
            <h2>عناوين IP المحظورة حاليًا</h2>
            <?php if (!empty($blocked_ips)) : ?>
                <p class="sityc-description">هذه قائمة الـ IPs التي تجاوزت الحد المسموح به وتم حظرها:</p>
                <table class="widefat fixed striped">
                    <thead>
                        <tr>
                            <th>عنوان IP</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($blocked_ips as $ip) : ?>
                        <tr>
                            <td><?php echo esc_html($ip); ?></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else : ?>
                <p class="sityc-description">لا توجد عناوين IP محظورة حاليًا 👌</p>
            <?php endif; ?>

            <form method="post" action="" style="margin-top:15px;">
                <?php wp_nonce_field('sityc_firewall_reset_ips', 'sityc_firewall_reset_nonce'); ?>
                <button type="submit" name="sityc_firewall_reset_ips" class="button sityc-btn-secondary">
                    🧹 مسح قائمة الـ IP المحظورة
                </button>
            </form>
        </div>

        <div class="sityc-footer-note">
            <strong>Sity Cloud Firewall</strong> يوفر طبقة حماية بسيطة للأحمال الزائدة، لكنه لا يغني عن أنظمة حماية متقدمة أو جدار ناري خارجي.
        </div>
    </div>

    <style>
        .sityc-firewall-wrap {
            direction: rtl;
            font-family: Tahoma, Arial, sans-serif;
        }
        .sityc-title {
            margin-bottom: 20px;
        }
        .sityc-card {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 16px;
            padding: 20px 24px;
            margin-bottom: 20px;
            box-shadow: 0 10px 25px rgba(15, 23, 42, 0.08);
            border: 1px solid rgba(148, 87, 235, 0.12);
            position: relative;
            overflow: hidden;
        }
        .sityc-card::before {
            content: '';
            position: absolute;
            inset: -40%;
            background: radial-gradient(circle at top left, rgba(148, 87, 235, 0.12), transparent 55%);
            opacity: 0.6;
            pointer-events: none;
        }
        .sityc-card h2 {
            margin-top: 0;
            margin-bottom: 8px;
        }
        .sityc-description {
            margin-top: 0;
            margin-bottom: 16px;
            color: #555;
        }
        .sityc-status-container {
            display: flex;
            gap: 16px;
            margin: 16px 0 20px;
            flex-wrap: wrap;
        }
        .sityc-radio {
            display: flex;
            align-items: center;
            gap: 8px;
            cursor: pointer;
        }
        .sityc-radio input[type="radio"] {
            transform: scale(1.1);
        }
        .sityc-radio-label {
            padding: 6px 12px;
            border-radius: 999px;
            font-size: 14px;
            border: 1px solid transparent;
        }
        .sityc-radio-label.sityc-on {
            background: rgba(34, 197, 94, 0.08);
            border-color: rgba(34, 197, 94, 0.4);
            color: #15803d;
        }
        .sityc-radio-label.sityc-off {
            background: rgba(239, 68, 68, 0.06);
            border-color: rgba(239, 68, 68, 0.4);
            color: #b91c1c;
        }
        .sityc-field-group {
            margin-bottom: 16px;
        }
        .sityc-field-group label {
            display: block;
            margin-bottom: 6px;
            font-weight: 600;
        }
        .sityc-field-group input[type="number"] {
            width: 180px;
            padding: 6px 8px;
            border-radius: 8px;
            border: 1px solid #cbd5e1;
        }
        .sityc-help {
            margin-top: 6px;
            font-size: 12px;
            color: #64748b;
        }
        .sityc-btn-primary {
            background: linear-gradient(135deg, #7c3aed, #a855f7);
            border: none;
            box-shadow: 0 8px 20px rgba(124, 58, 237, 0.35);
        }
        .sityc-btn-primary:hover {
            background: linear-gradient(135deg, #6d28d9, #9333ea);
        }
        .sityc-btn-secondary {
            background: #f9fafb;
            border-radius: 999px;
            border: 1px solid #e5e7eb;
            color: #374151;
        }
        .sityc-btn-secondary:hover {
            background: #e5e7eb;
        }
        .sityc-footer-note {
            font-size: 12px;
            color: #6b7280;
            margin-top: 8px;
        }
    </style>
    <?php
}

/**
 * حماية الواجهة الأمامية: عدّ عدد الطلبات في الجلسة وحظر الـ IP عند تجاوز الحد
 */
add_action('template_redirect', 'sityc_firewall_protection');
function sityc_firewall_protection() {
    // لو الحماية مقفولة .. خروج
    if (get_option('sityc_firewall_status', 'off') !== 'on') {
        return;
    }

    // لا تحظر الأدمن
    if (is_user_logged_in() && current_user_can('manage_options')) {
        return;
    }

    // بدء جلسة
    if (!session_id()) {
        @session_start();
    }

    // IP الزائر
    $visitor_ip = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : 'UNKNOWN';

    // عدّاد الطلبات
    if (!isset($_SESSION['sityc_request_count'])) {
        $_SESSION['sityc_request_count'] = 0;
    }
    $_SESSION['sityc_request_count']++;

    $request_limit = intval(get_option('sityc_firewall_request_limit', 20));
    if ($request_limit < 5) {
        $request_limit = 20;
    }

    // لو تعدى الحد المسموح
    if ($_SESSION['sityc_request_count'] > $request_limit) {

        // حفظ IP في القائمة
        $blocked_ips = get_option('sityc_blocked_ips', []);
        if (!is_array($blocked_ips)) {
            $blocked_ips = [];
        }
        if (!in_array($visitor_ip, $blocked_ips, true)) {
            $blocked_ips[] = $visitor_ip;
            update_option('sityc_blocked_ips', $blocked_ips);
        }

        // صفحة حظر بتصميم بسيط باسم Sity Cloud
        status_header(403);
        ?>
        <!DOCTYPE html>
        <html <?php language_attributes(); ?>>
        <head>
            <meta charset="<?php bloginfo('charset'); ?>">
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <title>تم حظر الوصول - Sity Cloud Firewall</title>
            <style>
                body {
                    margin: 0;
                    padding: 0;
                    font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
                    background: radial-gradient(circle at top, #ede9fe, #f9fafb);
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    min-height: 100vh;
                    direction: rtl;
                }
                .sityc-block-card {
                    background: rgba(255, 255, 255, 0.9);
                    backdrop-filter: blur(16px);
                    border-radius: 24px;
                    padding: 32px 28px;
                    max-width: 480px;
                    width: 90%;
                    text-align: center;
                    box-shadow: 0 20px 45px rgba(15, 23, 42, 0.18);
                    border: 1px solid rgba(148, 87, 235, 0.18);
                    position: relative;
                    overflow: hidden;
                }
                .sityc-block-card::before {
                    content: '';
                    position: absolute;
                    width: 220px;
                    height: 220px;
                    border-radius: 999px;
                    background: radial-gradient(circle, rgba(124, 58, 237, 0.22), transparent 60%);
                    top: -80px;
                    left: -60px;
                    opacity: 0.8;
                }
                .sityc-block-icon {
                    position: relative;
                    width: 70px;
                    height: 70px;
                    border-radius: 999px;
                    margin: 0 auto 16px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    background: radial-gradient(circle at top, #7c3aed, #a855f7);
                    color: #fff;
                    font-size: 32px;
                    box-shadow: 0 15px 35px rgba(124, 58, 237, 0.55);
                }
                .sityc-block-title {
                    position: relative;
                    font-size: 24px;
                    margin-bottom: 10px;
                    color: #111827;
                }
                .sityc-block-text {
                    position: relative;
                    font-size: 14px;
                    color: #4b5563;
                    margin-bottom: 16px;
                    line-height: 1.7;
                }
                .sityc-block-tag {
                    position: relative;
                    display: inline-flex;
                    align-items: center;
                    gap: 6px;
                    padding: 6px 12px;
                    border-radius: 999px;
                    background: rgba(243, 244, 246, 0.9);
                    font-size: 13px;
                    color: #6b7280;
                    margin-bottom: 6px;
                }
                .sityc-block-ip {
                    position: relative;
                    font-size: 13px;
                    color: #6b7280;
                }
                .sityc-block-brand {
                    position: relative;
                    font-size: 12px;
                    color: #7c3aed;
                    margin-top: 16px;
                }
            </style>
        </head>
        <body>
            <div class="sityc-block-card">
                <div class="sityc-block-icon">🛡️</div>
                <h1 class="sityc-block-title">تم حظر طلباتك مؤقتًا</h1>
                <p class="sityc-block-text">
                    نظام الحماية في <strong>Sity Cloud Firewall</strong> رصد عدد كبير من الطلبات المتتابعة من اتصالك،
                    لذلك تم إيقاف الوصول مؤقتًا لحماية الموقع من الضغط الزائد.
                </p>
                <div class="sityc-block-tag">
                    حماية مفعّلة عبر Sity Cloud
                </div>
                <div class="sityc-block-ip">
                    عنوان الاتصال الحالي: <?php echo esc_html($visitor_ip); ?>
                </div>
                <div class="sityc-block-brand">
                    بإدارة منصة استضافة المواقع <strong>Sity Cloud</strong>
                </div>
            </div>
        </body>
        </html>
        <?php
        exit;
    }
}

/**
 * بوب-أب إشعار الحماية في الفوتر بتصميم لايت مود / بنفسجي / زجاجي
 */
add_action('wp_footer', 'sityc_firewall_popup');
function sityc_firewall_popup() {
    if (get_option('sityc_firewall_status', 'off') !== 'on') {
        return;
    }

    // لا يظهر في لوحة التحكم
    if (is_admin()) {
        return;
    }
    ?>
    <div id="sityc-firewall-popup" class="sityc-firewall-popup" style="display:none;">
        <div class="sityc-firewall-inner">
            <div class="sityc-firewall-icon">🛡️</div>
            <div class="sityc-firewall-content">
                <div class="sityc-firewall-title">الموقع محمي بواسطة Sity Cloud Firewall</div>
                <div class="sityc-firewall-text">
                    يتم مراقبة حركة الزيارات لحظيًا لحماية موقعك من الهجمات والطلبات المريبة.
                </div>
            </div>
            <button type="button" class="sityc-firewall-close" aria-label="إغلاق">
                ✕
            </button>
        </div>
    </div>

    <style>
        .sityc-firewall-popup {
            position: fixed;
            bottom: 20px;
            left: 20px;
            z-index: 999999;
            direction: rtl;
            font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
        }
        .sityc-firewall-inner {
            display: flex;
            align-items: center;
            gap: 12px;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 18px;
            padding: 12px 14px;
            min-width: 260px;
            max-width: 360px;
            box-shadow: 0 18px 45px rgba(15, 23, 42, 0.25);
            border: 1px solid rgba(148, 87, 235, 0.35);
            backdrop-filter: blur(16px);
            transform: translateY(20px);
            opacity: 0;
            pointer-events: none;
            transition: opacity 0.35s ease, transform 0.35s ease;
        }
        .sityc-firewall-popup.sityc-visible .sityc-firewall-inner {
            opacity: 1;
            transform: translateY(0);
            pointer-events: auto;
        }
        .sityc-firewall-icon {
            width: 34px;
            height: 34px;
            border-radius: 999px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: radial-gradient(circle at top, #7c3aed, #a855f7);
            color: #fff;
            font-size: 20px;
            flex-shrink: 0;
            box-shadow: 0 8px 20px rgba(124, 58, 237, 0.5);
        }
        .sityc-firewall-content {
            flex: 1;
        }
        .sityc-firewall-title {
            font-size: 14px;
            font-weight: 700;
            color: #111827;
            margin-bottom: 2px;
        }
        .sityc-firewall-text {
            font-size: 12px;
            color: #4b5563;
            line-height: 1.6;
        }
        .sityc-firewall-close {
            border: none;
            background: transparent;
            color: #6b7280;
            cursor: pointer;
            font-size: 14px;
            padding: 2px 4px;
            border-radius: 999px;
            transition: background 0.2s ease, color 0.2s ease;
        }
        .sityc-firewall-close:hover {
            background: rgba(148, 87, 235, 0.08);
            color: #4c1d95;
        }

        @media (max-width: 600px) {
            .sityc-firewall-popup {
                left: 10px;
                right: 10px;
            }
            .sityc-firewall-inner {
                max-width: none;
            }
        }
    </style>

    <script>
        (function() {
            var popup = document.getElementById('sityc-firewall-popup');
            if (!popup) return;

            var inner = popup.querySelector('.sityc-firewall-inner');
            var closeBtn = popup.querySelector('.sityc-firewall-close');

            // إظهار البوب-أب بعد 8 ثواني
            setTimeout(function() {
                popup.classList.add('sityc-visible');
            }, 8000);

            if (closeBtn) {
                closeBtn.addEventListener('click', function() {
                    popup.classList.remove('sityc-visible');
                });
            }
        })();
    </script>
    <?php
}
